function ima_fil = ppb_gauss (ima_nse, sigma, ...
                              hW, hD, ...
                              alpha, T, ...
                              nbit, ima_est)
%PPB_GAUSS Iterative PPB filter for Gaussian noise
%   IMA_FIL = PPB_GAUSS(IMA_NSE, SIGMA, HW, HD, ALPHA, T, NBIT, IMA_EST)
%   denoises iteratively an image corrupted by additive white Gaussian
%   noise with the iterative Probabilistic Patch-Based (PPB)
%   filter described in "Iterative Weighted Maximum Likelihood
%   Denoising with Probabilistic Patch-Based Weights", written by
%   C.A. Deledalle, L. Denis and F. Tupin, IEEE Trans. on Image
%   Processing, vol. 18, no. 12, pp. 2661-2672, December 2009.
%   Please refer to this paper for a more detailed description of
%   the arguments. Note that this function ables also to treat
%   large images by preserving memory thanks to a block processing
%   on 1024x1024 subimages.
%
%       ARGUMENT DESCRIPTION:
%               IMA_NSE  - Noisy image
%               SIGMA    - Standard deviation of the Gaussian noise
%                          (default 20)
%               HW       - Half sizes of the search window width
%                          (default 10)
%               HD       - Half sizes of the  window width
%                          (default 3)
%               ALPHA    - Alpha-quantile parameters on the noisy image
%                          (default 0.92)
%               T        - filtering parameters on the estimated image
%                          (default 0.2)
%               NBIT     - numbers of iteration
%                          (default 4)
%               IMA_EST  - First noise-free image estimate (Optional).
%                          (default constant image)
%
%       OUTPUT DESCRIPTION:
%               IMA_FIL  - Fixed-point Filtered Image

    if nargin < 2
        sigma = 20;
    end
    if nargin < 3
        hW = 10;
    end
    if nargin < 4
        hD  = 3;
    end
    if nargin < 5
        alpha = 0.92;
    end
    if nargin < 6
        T = 0.2;
    end
    if nargin < 7
        nbit = 4;
    end
    if nargin < 8
        ima_est = ones(size(ima_nse));
    end

    W = 2 * hW + 1;
    D = 2 * hD + 1;
    h = quantile_gauss(D, alpha) * (4 * sigma^2) .* D.^2;

    T = T * sigma^2 ./ h .* D.^2;

    width = size(ima_nse, 1);
    height = size(ima_nse, 2);
    sw = 1024;
    sh = 1024;
    overlap = hW + hD;

    for l = 1:size(nbit, 2)
        for k = 1:nbit(l)
            for i = 0:(ceil(width / sw) - 1)
                for j = 0:(ceil(height / sh) - 1)
                    sx = 1 + i * sw;
                    ex = sw + i * sw;
                    sy = 1 + j * sh;
                    ey = sh + j * sh;
                    margesx = overlap(l);
                    margeex = overlap(l);
                    margesy = overlap(l);
                    margeey = overlap(l);
                    if ex > width
                        ex = width;
                    end
                    if ey > height
                        ey = height;
                    end
                    if sx - margesx < 1
                        margesx = 0;
                    end
                    if ex + margeex > width
                        margeex = 0;
                    end
                    if sy - margesy < 1
                        margesy = 0;
                    end
                    if ey + margeey > height
                        margeey = 0;
                    end

                    xrange = (sx - margesx):(ex + margeex);
                    yrange = (sy - margesy):(ey + margeey);

                    sub_ima_nse = ima_nse(xrange, yrange);
                    sub_ima_est = ima_est(xrange, yrange);

                    clear input;
                    input(:,:,1) = sub_ima_nse;
                    input(:,:,2) = sub_ima_est;

                    [sub_ima_fil] = ...
                        ppbGauss (input, ...
                                  hW(l), ...
                                  hD(l), ...
                                  h(l), T(l));

                    xrange = (1 + margesx):(ex - sx + 1 + margesx);
                    yrange = (1 + margesy):(ey - sy + 1 + margesy);

                    ima_fil(sx:ex, sy:ey, l) = sub_ima_fil(xrange, yrange);
                end
            end
            ima_est = ima_fil(:,:,l);
        end
    end
    l = size(ima_fil, 3);
    ima_fil = ima_fil(:,:,l);
end

function r = quantile(x, alpha) % SAS-5

       x = sort(x);
       N = size(x, 2);
       h = N * alpha + 0.5;
       if alpha == 0
           r = x(1);
           return;
       end
       if alpha == 1
           r = x(N);
           return;
       end
       r = (x(ceil(h - 0.5)) + x(floor(h + 0.5))) / 2;
end

function r = quantile_gauss(ws, alphas)

    for kw = 1:size(ws, 2);
        w = ws(kw);
        ima_nse = zeros(w * 256);

        ima_nse = ima_nse + randn(size(ima_nse));

        k = 1;
        for i = 1:(2*w):(size(ima_nse,1) - 2*w)
            for j = 1:w:(size(ima_nse,2) - w)
                sub_nse_1 = ima_nse(i:(i + w - 1), j:(j + w - 1));
                sub_nse_2 = ima_nse((i + w):(i + 2 * w - 1), j:(j + w - 1));

                lsl = (sub_nse_1 - sub_nse_2).^2 / 4;

                v(k) = mean(mean(lsl));
                k = k + 1;
            end
        end

        for q = 1:size(alphas, 2)
            r(q, kw) = quantile(v, alphas(q)) - mean(v);
        end
    end

end

